from typing import List, Optional, Tuple

from axelrod._strategy_utils import detect_cycle

from axelrod.action import Action

from axelrod.player import Player

C, D = Action.C, Action.D

def is_alternator(history: List[Action]) -> bool:
    for i in range(len(history) - 1):
        if history[i] == history[i + 1]:
            return False
    return True

class RandomHunter(Player):
    """A player who hunts for random players.

    Names:

    - Random Hunter: Original name by Karol Langner
    """

    name = "Random Hunter"
    classifier = {
        "memory_depth": float("inf"),  # Long memory
        "stochastic": False,
        "long_run_time": False,
        "inspects_source": False,
        "manipulates_source": False,
        "manipulates_state": False,
    }

    def __init__(self) -> None:
        self.countCC = 0
        self.countDD = 0
        super().__init__()

    def strategy(self, opponent: Player) -> Action:
        """
        A random player is unpredictable, which means the conditional frequency
        of cooperation after cooperation, and defection after defections, should
        be close to 50%... although how close is debatable.
        """
        # Update counts
        if len(self.history) > 1:
            if self.history[-2] == C and opponent.history[-1] == C:
                self.countCC += 1
            if self.history[-2] == D and opponent.history[-1] == D:
                self.countDD += 1

        n = len(self.history)
        if n > 10:
            probabilities = []
            if self.cooperations > 5:
                probabilities.append(self.countCC / self.cooperations)
            if self.defections > 5:
                probabilities.append(self.countDD / self.defections)
            if probabilities and all(
                [abs(p - 0.5) < 0.25 for p in probabilities]
            ):
                return D
        return C